#!/usr/bin/python
#
# iwlistparse.py
# Hugo Chargois - 17 jan. 2010 - v.0.1
# https://bbs.archlinux.org/viewtopic.php?pid=761899
# Parses the output of iwlist scan into a table

import sys

def get_name(cell):
    return matching_line(cell,"ESSID:")[1:-1]

def get_quality(cell):
    quality = matching_line(cell,"Quality=").split()[0].split('/')
    return int(round(float(quality[0]) / float(quality[1]) * 100))

def get_channel(cell):
    return int(matching_line(cell,"Channel:"))

def get_encryption(cell):
    enc=""
    if matching_line(cell,"Encryption key:") == "off":
        enc="Open"
    else:
        for line in cell:
            matching = match(line,"IE:")
            if matching!=None:
                if detect(matching, "WPA Version"):
                    enc="WPA"
                if detect(matching, "IEEE 802.11i/WPA2 Version"):
                    enc="WPA2"
        if enc=="":
            enc="WEP"
    return enc

def get_encryption_detail(cell):
    enc=""
    for line in cell:
        matching = match(line,"Group Cipher : ")
        if matching!=None:
            enc=matching
        matching = match(line,"Authentication Suites (1) : ")
        if matching!=None:
            if matching == "802.1x":
                enc=enc+"/"+"EAP"
            else:
                enc=enc+"/"+matching
    return enc

def get_address(cell):
    return matching_line(cell,"Address: ")

# Here's a dictionary of rules that will be applied to the description of each
# cell. The key will be the name of the column in the table. The value is a
# function defined above.

rules={"Name":get_name,
       "Quality":get_quality,
       "Channel":get_channel,
       "Encryption":get_encryption,
       "Address":get_address,
       "Detail":get_encryption_detail,
       }

# Here you can choose the way of sorting the table. sortby should be a key of
# the dictionary rules.

def sort_cells(cells):
    sortby = "Quality"
    reverse = True
    cells.sort(None, lambda el:el[sortby], reverse)

# You can choose which columns to display here, and most importantly in what order. Of
# course, they must exist as keys in the dict rules.

columns=["Name","Address","Quality","Channel","Encryption", "Detail"]

# Below here goes the boring stuff. You shouldn't have to edit anything below
# this point

def matching_line(lines, keyword):
    """Returns the first matching line in a list of lines. See match()"""
    for line in lines:
        matching=match(line,keyword)
        if matching!=None:
            return matching
    return None

def match(line,keyword):
    """If the first part of line (modulo blanks) matches keyword,
    returns the end of that line. Otherwise returns None"""
    line=line.lstrip()
    length=len(keyword)
    if line[:length] == keyword:
        return line[length:]
    else:
        return None

def detect(line, keyword):
    line=line.lstrip()
    length=len(keyword)
    if line[:length] == keyword:
        return 1
    else:
        return 0


def parse_cell(cell):
    """Applies the rules to the bunch of text describing a cell and returns the
    corresponding dictionary"""
    parsed_cell={}
    for key in rules:
        rule=rules[key]
        parsed_cell.update({key:rule(cell)})
    return parsed_cell

def print_table(table):
    # widths=map(max,map(lambda l:map(len,l),zip(*table))) #functional magic

    justified_table = []
    for line in table:
        justified_line=[]
        for i,el in enumerate(line):
            justified_line.append(str(el).ljust(30))
            # justified_line.append(el.ljust(widths[i]+2))
        justified_table.append(justified_line)
    
    for line in justified_table:
        for el in line:
            print el,
        print

def get_networks(list):
    cells=[[]]
    parsed_cells=[]

    for line in list:
        cell_line = match(line,"Cell ")
        if cell_line != None:
            cells.append([])
            line = cell_line[-27:]
        cells[-1].append(line.rstrip())

    cells=cells[1:]

    for cell in cells:
        parsed_cells.append(parse_cell(cell))

    sort_cells(parsed_cells)

    return parsed_cells

